/*
 * linux/arch/arm/mach-uniphier/globaltimer.c
 *
 * Copyright (C) 2011 Panasonic Corporation
 * All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 */
#include <linux/init.h>
#include <linux/io.h>
#include <linux/clockchips.h>

#include <mach/hardware.h>

#define GLOBAL_TIMER_START		(UNIPHIER_PRIVATE_IO_START + 0x200)
#define GLOBAL_TIMER_LOWER_COUNTER	0x0
#define GLOBAL_TIMER_UPPER_COUNTER	0x4
#define GLOBAL_TIMER_CONTROL		0x8

static cycle_t uniphier_get_cycles(struct clocksource *cs)
{
	return inl(GLOBAL_TIMER_START + GLOBAL_TIMER_LOWER_COUNTER);
}

static struct clocksource clocksource_globaltimer = {
	.name	= "global timer",
	.rating	= 200,
	.read	= uniphier_get_cycles,
	.mask	= CLOCKSOURCE_MASK(32),
	.flags	= CLOCK_SOURCE_IS_CONTINUOUS,
};

extern void clocksource_set_clock(struct clocksource *cs, unsigned int clock);

void start_globaltimer(void)
{
	/* setup global timer as free-running clocksource */
	outl(0, GLOBAL_TIMER_START + GLOBAL_TIMER_CONTROL);
	outl(0, GLOBAL_TIMER_START + GLOBAL_TIMER_LOWER_COUNTER);
	outl(0, GLOBAL_TIMER_START + GLOBAL_TIMER_UPPER_COUNTER);
	outl(1, GLOBAL_TIMER_START + GLOBAL_TIMER_CONTROL);
}

static unsigned long lower_counter;
static unsigned long upper_counter;

void stop_globaltimer(void)
{
	/* setup global timer as free-running clocksource */
	outl(0, GLOBAL_TIMER_START + GLOBAL_TIMER_CONTROL);
	lower_counter = inl(GLOBAL_TIMER_START + GLOBAL_TIMER_LOWER_COUNTER);
	upper_counter = inl(GLOBAL_TIMER_START + GLOBAL_TIMER_UPPER_COUNTER);
}

void restart_globaltimer(void)
{
	/* setup global timer as free-running clocksource */
	outl(0, GLOBAL_TIMER_START + GLOBAL_TIMER_CONTROL);
	outl(lower_counter, GLOBAL_TIMER_START + GLOBAL_TIMER_LOWER_COUNTER);
	outl(upper_counter, GLOBAL_TIMER_START + GLOBAL_TIMER_UPPER_COUNTER);
	outl(1, GLOBAL_TIMER_START + GLOBAL_TIMER_CONTROL);
}

void __init globaltimer_clocksource_init(void)
{
	start_globaltimer();
	clocksource_set_clock(&clocksource_globaltimer, UNIPHIER_GLOBAL_TIMER_SRC);

	clocksource_register(&clocksource_globaltimer);
}

/*
 * This is the UniPhier sched_clock implementation.  This has
 * a resolution of 20ns, and a maximum value of about 585 years!
 */
unsigned long long sched_clock(void)
{
	union {
		unsigned long long ll;
		struct {
			unsigned long lower;
			unsigned long upper;
		} l;
	} v;
	unsigned long upper0;

	do {
		upper0    = inl(GLOBAL_TIMER_START + GLOBAL_TIMER_UPPER_COUNTER);
		v.l.lower = inl(GLOBAL_TIMER_START + GLOBAL_TIMER_LOWER_COUNTER);
		v.l.upper = inl(GLOBAL_TIMER_START + GLOBAL_TIMER_UPPER_COUNTER);
	} while (v.l.upper != upper0);

	/* convert to nanosec units: 1E9 / UNIPHIER_GLOBAL_TIMER_SRC */
	v.ll *= 1000000000 / UNIPHIER_GLOBAL_TIMER_SRC;

	return v.ll;
}
